module net.BurtonRadons.dig.platform.editText;

private import net.BurtonRadons.dig.platform.control;
private import net.BurtonRadons.dig.platform.base;

//private import std.c.windows.windows;


/** A widget that provides a std.math.single-line text editor.
  * Whenever the text is changed, the onChange dispatcher
  * is notified.  This dispatcher accepts:
  *
  * @code
  * void delegate ();
  * void delegate (char [] text);
  * @endcode
  */
class EditText : public Control
{
    private import net.BurtonRadons.dig.common.expressionEvaluator;
    private import net.BurtonRadons.dig.platform.windows;
    private import net.BurtonRadons.dig.platform.frame;
    private const wchar [] digPlatformClassName = "digPlatformEditText";

    /** Handles the EditText.onChange dispatching. */
    struct ChangeDispatcher
    {
        typedef void delegate () dMethodA; /**< The form of a registered method. */
        typedef void delegate (char [] text) dMethodB; /**< Explicit form. */

        dMethodA [] methoda; /**< The list of methods called by notify. */
        dMethodB [] methodb; /**< Second list of methods called by notify. */

        /** Add a method to the list and enable dropping. */
        void add (dMethodA method)
        {
            methoda ~= method;
        }

        /** Add a second-form method to the list and enable dropping. */
        void add (dMethodB method)
        {
            methodb ~= method;
        }

        /** Notify the methods that an event has occured. */
        void notify (char [] text)
        {
            for (dMethodA *m = methoda, n = m + methoda.length; m < n; m ++)
                (*m) ();
            for (dMethodB *m = methodb, n = m + methodb.length; m < n; m ++)
                (*m) (text);
        }

        /** Remove all methods. */
        void empty ()
        {
            methoda = null;
            methodb = null;
        }
    }

    ChangeDispatcher onChange; /**< Notified when the edit text has been modified. */

    /** Setup the expression evaluator and the window class. */
    static this ()
    {
        digPlatformEvaluator = new ExpressionEvaluator ();
        digPlatformEvaluator.predefineMath ();
        
        GetClassInfoA (hinst, "Edit", &digPlatformBaseClass);
        
        digPlatformItemClass = digPlatformBaseClass;
        digPlatformItemClass.lpszClassName = wtoStringz (digPlatformClassName);
        digPlatformItemClass.lpfnWndProc = &digPlatformWindowProc;
        std.c.windows.windows.RegisterClassA (&digPlatformItemClass);
    }

    /** Register with the parent and create the editor. */
    this (Control parent)
    {
        super (parent);
        digPlatformStyle |= std.c.windows.windows.WS_VISIBLE | std.c.windows.windows.WS_CHILD | std.c.windows.windows.ES_AUTOHSCROLL | std.c.windows.windows.ES_WANTRETURN;
        digPlatformHWNDCreate (WS_EX_CLIENTEDGE, digPlatformClassName, null, digPlatformStyle, (_HANDLE) 0);
        digPlatformSuggestWidth = 20;
        digPlatformSuggestHeight = 20;
    }

    /** Set text. */
    void text (char [] string)
    {
        digPlatformSetText (string);
    }

    /** Get text. */
    char [] text ()
    {
        return digPlatformGetText ();
    }

    /** Get text as floating-point value, return whether it worked.
      * This handles all forms of float, all input forms of D, and
      * common expressions (+, -, *, /, and **), and bracket
      * nesting.
      */
    bit getFloat (out float value)
    {
        return digPlatformEvaluator.eval (this.text (), value);
    }

    /** If set, put a border around the control (the default is false) */
    void bordered (bit value)
    {
        digPlatformSetStyle (WS_BORDER, value);
    }

    /** Set whether this is active (true) or grayed and inactive (false). */
    void enabled (bit value)
    {
        EnableWindow (digPlatformHWND, value);
    }
    
    /** Set whether this is a multiple-line edit box (true) or a std.math.single line editor (false). */
    void multiline (bit value)
    {
        digPlatformSetRecreateStyle (std.c.windows.windows.ES_MULTILINE, value);
        digPlatformSetRecreateStyle (std.c.windows.windows.WS_VSCROLL, value);
        digPlatformSetRecreateStyle (std.c.windows.windows.ES_AUTOHSCROLL, !value);
        digPlatformSetRecreateStyle (std.c.windows.windows.ES_WANTRETURN, !value);
        digPlatformSetStyle (std.c.windows.windows.WS_BORDER, value);
        vscroll (value);
    }
    
/+
#ifdef DoxygenMustSkipThis
+/

    static ExpressionEvaluator digPlatformEvaluator; // The expression evaluator used with getFloat.
    static _WNDCLASS digPlatformBaseClass; // The "Edit" class this uses as a base.
    static _WNDCLASS digPlatformItemClass; // The class that we actually use.
    
    override void digPlatformCommand (int code, int id)
    {
        switch (code)
        {
            case std.c.windows.windows.EN_CHANGE:
                onChange.notify (text ());
                break;

            default:
                break;
        }

        /*if (notificationNameBase (code) === null)
            printf ("%d %d\n", code, id);*/
    }

    char [] digPlatformNotificationName (int code)
    {
        switch (code)
        {
            case EN_CHANGE: return "EN_CHANGE";
                /* Sent when the user has taken an action that may have
                   altered text in an edit control. Unlike the EN_UPDATE
                   notification message, this notification message is sent
                   after the system updates the screen. */
            case EN_KILLFOCUS: return "EN_KILLFOCUS";
                /* Sent when an edit control loses the keyboard focus. */
            case EN_SETFOCUS: return "EN_SETFOCUS"; /* Sent when an edit control receives the keyboard focus. */
            case EN_UPDATE: return "EN_UPDATE";
                /* Sent when an edit control is about to redraw itself.
                   This notification message is sent after the control has
                   formatted the text, but before it displays the text. This
                   makes it possible to resize the edit control window, if
                   necessary. */
            default: return fmt ("(%d)", code);
        }
    }

    /* Get the current text limit. */
    int digPlatformGetLimitText () { return SendMessageA (digPlatformHWND, EM_GETLIMITTEXT, 0, 0); }

    /* Set whether this is a read-only box. */
    void digPlatformSetReadOnly (bit value) { SendMessageA (digPlatformHWND, EM_SETREADONLY, value ? 1 : 0, 0); }
    
    static extern (Windows)
    _LRESULT digPlatformWindowProc (_HWND hwnd, _UINT message, _WPARAM wparam, _LPARAM lparam)
    {
        _HANDLE wparamHandle = (_HANDLE) wparam;
        Control control = digPlatformHWNDToControl [hwnd];
        EditText item = cast (EditText) control;
        _LRESULT result;
        
        switch (message)
        {
            case WM_KEYDOWN:
                if ((wparam >= VK_A && wparam <= VK_Z)
                 || (wparam >= VK_0 && wparam <= VK_9))
                    goto fallback;
            case WM_KEYUP:
            case WM_CHAR:
                result = CallWindowProcA (digPlatformBaseClass.lpfnWndProc, hwnd, message, wparam, lparam);
                Frame.digPlatformWindowProc (hwnd, message, wparam, lparam);
                break;
            
            default:
            fallback:
                result = CallWindowProcA (digPlatformBaseClass.lpfnWndProc, hwnd, message, wparam, lparam);
                break;
        }
        
        return result;
    }
    
/+
#endif
+/
}
